from apm import *

## Linear model of a Boeing 747
#  Level flight at 40,000 ft elevation
#  Velocity at 774 ft/sec (0.80 Mach)

# States
#  u - uw (ft/sec) - horizontal velocity - horizontal wind
#  w - ww (ft/sec) - vertical velocity - vertical wind
#  q (crad/sec) - angular velocity
#  theta (crad) - angle from horizontal
# note: crad = 0.01 rad

# Inputs
#  e - elevator
#  t - throttle

# Outputs
#  u - uw (ft/sec) - horizontal airspeed
#  hdot = -w + u0 * theta with u0 = 774 ft/sec

## continuous time simulation
s = 'http://byu.apmonitor.com'
a = '747c'

# elevator step
apm(s,a,'clear all')
apm_load(s,a,'747c.apm')
csv_load(s,a,'control.csv')
apm_option(s,a,'nlc.imode',6)
apm_option(s,a,'nlc.nodes',3)

# controller setup
# manipulated variables
apm_info(s,a,'MV','u[1]')
apm_info(s,a,'MV','u[2]')
# controlled variables
apm_info(s,a,'CV','y[1]')
apm_info(s,a,'CV','y[2]')

## MV tuning
# lower and upper bounds for elevator pitch
apm_option(s,a,'u[1].lower',-5)
apm_option(s,a,'u[1].upper',5)
# lower and upper bounds for thrust
apm_option(s,a,'u[2].lower',-5)
apm_option(s,a,'u[2].upper',5)
# delta MV movement cost
apm_option(s,a,'u[1].dcost',1)
apm_option(s,a,'u[2].dcost',1)

## CV tuning
# tau = first order time constant for trajectories
apm_option(s,a,'y[1].tau',5)
apm_option(s,a,'y[2].tau',8)
# tr_init = 0 (dead-band)
#         = 1 (first order trajectory)
#         = 2 (first order traj, re-center with each cycle)
apm_option(s,a,'y[1].tr_init',2)
apm_option(s,a,'y[2].tr_init',2)
# targets (dead-band needs upper and lower values)
# SPHI = upper set point
# SPLO = lower set point
apm_option(s,a,'y[1].sphi',-8.5)
apm_option(s,a,'y[1].splo',-9.5)
apm_option(s,a,'y[2].sphi',5.4)
apm_option(s,a,'y[2].splo',4.6)

# turn on MVs / CVs (0 = off, 1 = on)
apm_option(s,a,'u[1].status',1)
apm_option(s,a,'u[2].status',1)
apm_option(s,a,'y[1].status',1)
apm_option(s,a,'y[2].status',1)

# solve
output = apm(s,a,'solve')
print(output)

# retrieve solution
z = apm_sol(s,a)

# open web-viewer
apm_web(s,a)

## plot results
import matplotlib.pyplot as plt
plt.figure(1)
plt.subplot(311)
plt.plot(z['time'],z['u[1]'],'r-',linewidth=2.0)
plt.plot(z['time'],z['u[2]'],'k:',linewidth=2.0)
plt.legend(['Elevator','Thrust'])
plt.ylabel(['MV Action'])

plt.subplot(312)
plt.plot(z['time'],z['y[1]'],'b:',linewidth=2.0)
plt.plot(z['time'],z['y[1].tr_hi'],'k-')
plt.plot(z['time'],z['y[1].tr_lo'],'k-')
plt.legend(['Air Speed','Upper Trajectory','Lower Trajectory'])

plt.subplot(313)
plt.plot(z['time'],z['y[2]'],'g--',linewidth=2.0)
plt.plot(z['time'],z['y[2].tr_hi'],'k-')
plt.plot(z['time'],z['y[2].tr_lo'],'k-')
plt.legend(['Climb Rate','Upper Trajectory','Lower Trajectory'])

plt.show()
