clear all; close all; clc

addpath('apm')

%% Linear model of a Boeing 747
%  Level flight at 40,000 ft elevation
%  Velocity at 774 ft/sec (0.80 Mach)

% States
%  u - uw (ft/sec) - horizontal velocity - horizontal wind
%  w - ww (ft/sec) - vertical velocity - vertical wind
%  q (crad/sec) - angular velocity
%  theta (crad) - angle from horizontal
% note: crad = 0.01 rad

% Inputs
%  e - elevator
%  t - throttle

% Outputs
%  u - uw (ft/sec) - horizontal airspeed
%  hdot = -w + u0 * theta with u0 = 774 ft/sec

%% continuous time simulation
s = 'http://byu.apmonitor.com';
%s = 'http://127.0.0.1';
a = '747c';

% elevator step
apm(s,a,'clear all');
apm_load(s,a,'747c.apm');
csv_load(s,a,'control.csv');
apm_option(s,a,'nlc.imode',6);
apm_option(s,a,'nlc.nodes',3);

% controller setup
% manipulated variables
apm_info(s,a,'MV','u[1]');
apm_info(s,a,'MV','u[2]');
% controlled variables
apm_info(s,a,'CV','y[1]');
apm_info(s,a,'CV','y[2]');

%% MV tuning
% lower and upper bounds for elevator pitch
apm_option(s,a,'u[1].lower',-5);
apm_option(s,a,'u[1].upper',5);
% lower and upper bounds for thrust
apm_option(s,a,'u[2].lower',-5);
apm_option(s,a,'u[2].upper',5);
% delta MV movement cost
apm_option(s,a,'u[1].dcost',1);
apm_option(s,a,'u[2].dcost',1);

%% CV tuning
% tau = first order time constant for trajectories
apm_option(s,a,'y[1].tau',5);
apm_option(s,a,'y[2].tau',8);
% tr_init = 0 (dead-band)
%         = 1 (first order trajectory)
%         = 2 (first order traj, re-center with each cycle)
apm_option(s,a,'y[1].tr_init',2);
apm_option(s,a,'y[2].tr_init',2);
% targets (dead-band needs upper and lower values)
% SPHI = upper set point
% SPLO = lower set point
apm_option(s,a,'y[1].sphi',-8.5);
apm_option(s,a,'y[1].splo',-9.5);
apm_option(s,a,'y[2].sphi',5.4);
apm_option(s,a,'y[2].splo',4.6);

% turn on MVs / CVs (0 = off, 1 = on)
apm_option(s,a,'u[1].status',1);
apm_option(s,a,'u[2].status',1);
apm_option(s,a,'y[1].status',1);
apm_option(s,a,'y[2].status',1);

% solve
apm(s,a,'solve')

% retrieve solution
sol = apm_sol(s,a); z = sol.x;

% open web-viewer
apm_web(s,a);

%% plot results
figure(1)
subplot(3,1,1)
plot(z.time,z.u1,'r-','LineWidth',2)
hold on
plot(z.time,z.u2,'k:','LineWidth',2)
legend('Elevator','Thrust')
ylabel('MV Action')

subplot(3,1,2)
plot(z.time,z.y1,'b:','LineWidth',2)
hold on
plot(z.time,z.y1tr_hi,'k-','LineWidth',1)
plot(z.time,z.y1tr_lo,'k-','LineWidth',1)
legend('Air Speed')

subplot(3,1,3)
plot(z.time,z.y2,'g--','LineWidth',2)
hold on
plot(z.time,z.y2tr_hi,'k-','LineWidth',1)
plot(z.time,z.y2tr_lo,'k-','LineWidth',1)
legend('Climb Rate')
