import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import odeint

s = 'http://byu.apmonitor.com'
a = 'mpc'

from apm import *

# define CSTR model
def cstr(x,t,u,Tf,Caf):
    # Inputs (3):
    # Temperature of cooling jacket (K)
    Tc = u
    # Tf = Feed Temperature (K)
    # Caf = Feed Concentration (mol/m^3)

    # States (2):
    # Concentration of A in CSTR (mol/m^3)
    Ca = x[0]
    # Temperature in CSTR (K)
    T = x[1]

    # Parameters:
    # Volumetric Flowrate (m^3/sec)
    q = 100
    # Volume of CSTR (m^3)
    V = 100
    # Density of A-B Mixture (kg/m^3)
    rho = 1000
    # Heat capacity of A-B Mixture (J/kg-K)
    Cp = 0.239
    # Heat of reaction for A->B (J/mol)
    mdelH = 5e4
    # E - Activation energy in the Arrhenius Equation (J/mol)
    # R - Universal Gas Constant = 8.31451 J/mol-K
    EoverR = 8750
    # Pre-exponential factor (1/sec)
    k0 = 7.2e10
    # U - Overall Heat Transfer Coefficient (W/m^2-K)
    # A - Area - this value is specific for the U calculation (m^2)
    UA = 5e4
    # reaction rate
    rA = k0*np.exp(-EoverR/T)*Ca

    # Calculate concentration derivative
    dCadt = q/V*(Caf - Ca) - rA
    # Calculate temperature derivative
    dTdt = q/V*(Tf - T) \
            + mdelH/(rho*Cp)*rA \
            + UA/V/rho/Cp*(Tc-T)
    
    # Return xdot:
    xdot = np.zeros(2)
    xdot[0] = dCadt
    xdot[1] = dTdt
    return xdot

def mpc_init():
    fid = open('model.apm','w')
    fid.write('Constants \n')
    fid.write(' Tc_ss = 300 \n')
    fid.write(' T_ss = 324.0 \n')
    fid.write('Parameters \n')
    fid.write(' Tc = Tc_ss , <350 , >250 \n')
    fid.write(' K = 1.3 \n')
    fid.write(' tau = 0.9 \n')
    fid.write('Variables \n')
    fid.write(' T = T_ss \n')
    fid.write('Equations \n')
    fid.write(' tau * $T = -(T-T_ss) + K * (Tc-Tc_ss) \n')
    fid.close()

    t = np.linspace(0,1,21)
    fid = open('data.csv','w')
    fid.write('time \n')
    for ti in t:
        fid.write('{}\n'.format(ti))
    fid.close()

    apm(s,a,'clear all')
    apm_load(s,a,'model.apm')
    csv_load(s,a,'data.csv')

    apm_info(s,a,'MV','Tc')
    apm_info(s,a,'CV','T')

    apm_option(s,a,'Tc.fstatus',0)
    apm_option(s,a,'Tc.status',1)

    apm_option(s,a,'T.fstatus',1)
    apm_option(s,a,'T.status',1)
    apm_option(s,a,'T.sphi',300.1)
    apm_option(s,a,'T.splo',299.9)
    apm_option(s,a,'T.tau',0.1)
    apm_option(s,a,'T.tr_init',1)

    apm_option(s,a,'nlc.imode',6)
    apm_option(s,a,'nlc.web_plot_freq',1)

    msg = 'Successful initialization'
    return msg

# initialize application
mpc_init()

def mpc(T):
    apm_meas(s,a,'T',T)
    apm(s,a,'solve')
    Tc = apm_tag(s,a,'Tc.newval')
    
    return Tc


# Steady State Initial Conditions for the States
Ca_ss = 0.87725294608097
T_ss = 324.475443431599
x0 = np.empty(2)
x0[0] = Ca_ss
x0[1] = T_ss

# Steady State Initial Condition
u_ss = 300.0
# Feed Temperature (K)
Tf = 350
# Feed Concentration (mol/m^3)
Caf = 1

# Time Interval (min)
t = np.arange(0,5,0.05)

# Store results for plotting
Ca = np.ones(len(t)) * Ca_ss
T = np.ones(len(t)) * T_ss
u = np.ones(len(t)) * u_ss


# storage for recording values
op = np.ones(len(t))*u_ss  # controller output
pv = np.zeros(len(t))  # process variable
sp = np.zeros(len(t))  # set point
sp[0:50] = 300.0
sp[50:] = 320.0

# Upper and Lower limits on OP
op_hi = 350.0
op_lo = 250.0

pv[0] = T_ss
# loop through time steps    
for i in range(len(t)-1):
    # Tc = mpc(T)
    op[i] = mpc(T[i])
    if i==5:
        apm_web(s,a)
    if i==49:
        # change set point
        apm_option(s,a,'T.sphi',320.1)
        apm_option(s,a,'T.splo',319.9)

    ts = [t[i],t[i+1]]
    u[i+1] = op[i]
    y = odeint(cstr,x0,ts,args=(u[i+1],Tf,Caf))
    Ca[i+1] = y[-1][0]
    T[i+1] = y[-1][1]
    x0[0] = Ca[i+1]
    x0[1] = T[i+1]
    pv[i+1] = T[i+1]


op[len(t)-1] = op[len(t)-2]

# Construct results and save data file
# Column 1 = time
# Column 2 = cooling temperature
# Column 3 = reactor temperature
data = np.vstack((t,u,T)) # vertical stack
data = data.T             # transpose data
np.savetxt('data_doublet.txt',data,delimiter=',')
    
# Plot the results
plt.figure(1)
plt.subplot(4,1,1)
plt.plot(t,u,'b--',linewidth=3)
plt.ylabel('Cooling T (K)')
plt.legend(['Jacket Temperature'],loc=1)

plt.subplot(4,1,2)
plt.plot(t,Ca,'g-',linewidth=3)
plt.ylabel('Ca (mol/L)')
plt.legend(['Reactor Concentration'],loc=1)

plt.subplot(4,1,3)
plt.plot(t,T,'k:',linewidth=3,label='Reactor Temperature')
plt.plot(t,sp,'r--',linewidth=2,label='Set Point')
plt.ylabel('T (K)')
plt.xlabel('Time (min)')
plt.legend()

plt.subplot(4,1,4)
plt.plot(t,op,'r--',linewidth=3,label='Controller Output (OP)')
plt.legend()
plt.ylabel('Output')

plt.show()
