"""
Ch En 593R - Dynamic Optimization
Control and Optimization of a Zero-Carbon Power Grid

Written by: 
    Tyrel Hess
    Derek Prestwich
    Cameron Price
"""
##########################################################################
# Import relevant packages
import numpy as np
from gekko import GEKKO
import matplotlib.pyplot as plt
from scipy.integrate import odeint
import pandas as pd
from scipy.interpolate import interp1d

##########################################################################
# Demand data file import
filename = 'powergrid_week.csv'

##########################################################################
# Read demand data into arrays
data = pd.read_csv(filename)
time0 = data['hour'].values
power0 = data['nuc_power'].values/170.0
tot_power0 = data['tot_power'].values/170.0
solar_power0 = data['solar_power'].values/170.0
wind_power0 = data['wind_power'].values/170.0
time0 = time0-1

##########################################################################
# Interpolate demand data to match python data resolution
res = 4 #resolution multiplier

fnuc = interp1d(time0,power0)
fsol = interp1d(time0,solar_power0)
fwin = interp1d(time0,wind_power0)
ftot = interp1d(time0,tot_power0)

time = np.linspace(0,time0[-1],len(time0)*res)

power = np.zeros(len(time))
tot_power = np.zeros(len(time))
solar_power = np.zeros(len(time))
wind_power = np.zeros(len(time))

for i in range(len(time)):
    power[i] = fnuc(time[i])
    tot_power[i] = ftot(time[i])
    solar_power[i] = fsol(time[i])
    wind_power[i] = fwin(time[i])
    
##########################################################################
# Define n to set array lengths throughout the code
n = len(time)

##########################################################################
# Create a digital twin to be calculated with odeint
class save(): # Create a class to save values from the ODEINT model
    U = 0
    
def reactor(x,t,u,mult):
	# Parameters
	Beta = 0.007108
	Betas = [0.000216, 0.001416, 0.001349, 0.00218, 0.00095, 0.000322]

	Lambdas = [0.0125, 0.0308, 0.1152, 0.3109, 1.24, 3.3287] # s^-1

	l = 5e-4 # s

	rho_0 = -0.009108
	b = 0.00025

	alpha_f = -2e-5 # C^-1
	alpha_c = -5e-5 # C^-1

	Tin = 292 # C, being held constant by secondary loop

	# Reactor variables
	# Tout is approximately equal to Tavg

	#Create differentiable variables for odeint model
	P = x[0]
	C1 = x[1]
	C2 = x[2]
	C3 = x[3]
	C4 = x[4]
	C5 = x[5]
	C6 = x[6]
	Tf = x[7]
	Tavg = x[8]

	# Parameters
	A = 6.314e3 # m^2
	Mf = 122789.605 # kg
	Cpf = 0.0002999667 # MW * s * kg^-1 * C^-1 at 700 C for UO2
	J = 38 # MW/%
	Mc = 1.917e5 # kg
	mdot = 65.9e6 / 3600 # kg/s, alternative of 800 kg/s
	Cpc = 0.00571 # MW * s * kg^-1 * C^-1 at @ 298 C for water
	Cv = 0.0301 # flow coefficient
	De = 0.01297 # m
	Kc = 0.00000055 # MW/m*K
	mu = 0.000068 # Pa*s
	density = 1000 # kg/m^3

	# get U value
	# get velocity
	v = (mdot / 1000) / (np.pi * De**2 / 43) / 100000
	Re = De * v * density / mu
	Pr = Cpc * mu / Kc
	U = Cv * Re**0.8 * Pr**0.4 * (Kc / De) * mult
	save.U = U

	# reactivity
	rho = rho_0 + alpha_f * Tf + alpha_c * Tavg + b * u

	# differential equations
	dPdt = (rho - Beta) * P / l + (Lambdas[0] * C1 + Lambdas[1] * C2 + Lambdas[2] * C3 + Lambdas[3] * C4 + Lambdas[4] * C5 + Lambdas[5] * C6)
	dC1dt = (Betas[0] / l) * P - Lambdas[0] * C1
	dC2dt = (Betas[1] / l) * P - Lambdas[1] * C2
	dC3dt = (Betas[2] / l) * P - Lambdas[2] * C3
	dC4dt = (Betas[3] / l) * P - Lambdas[3] * C4
	dC5dt = (Betas[4] / l) * P - Lambdas[4] * C5
	dC6dt = (Betas[5] / l) * P - Lambdas[5] * C6
	dTfdt = -U * A / (Mf * Cpf) * (Tf - Tavg) + J * P / (Mf * Cpf)
	dTavgdt = U * A / (Mc * Cpc) * (Tf - Tavg) - (mdot / Mc) * (Tavg - Tin)

	diff_eqns = [dPdt, dC1dt, dC2dt, dC3dt, dC4dt, dC5dt, dC6dt, dTfdt, dTavgdt]
	return diff_eqns

##########################################################################  
# Set up parameters for GEKKO models
# Parameters
Beta = 0.007108
Betas = [0.000216, 0.001416, 0.001349, 0.00218, 0.00095, 0.000322]

Lambda = 0.078 # s^-1
Lambdas = [0.0125, 0.0308, 0.1152, 0.3109, 1.24, 3.3287] # s^-1

l = 5e-4 # s

LT = 50 # potential efficiency of 0.34

rho_0 = -0.009108

b = 0.00025

alpha_f = -2e-5 # C^-1
alpha_c = -5e-5 # C^-1

Tin = 292 # C

A = 6.314e3 # m^2
Mf = 122789.605 # kg
Cpf = 0.0002999667 # MW * s * kg^-1 * C^-1 at 700 C for UO2
J = 38 # MW/%
Mc = 1.917e5 # kg
mdot = 65.9e6 / 3600 # kg/s, alternative of 800 kg/s
Cpc = 0.00571 # MW * s * kg^-1 * C^-1 at @ 298 C for water
Cv = 0.0301 # flow coefficient
De = 0.01297 # m
Kc = 0.00000055 # MW/m*K
mu = 0.000068 # Pa*s
density = 1000 # kg/m^3

v = (mdot / 1000) / (np.pi * De**2 / 43) / 100000
Re = De * v * density / mu
Pr = Cpc * mu / Kc
U = Cv * Re**0.8 * Pr**0.4 * (Kc / De) * 0.03

Ctot = 1e3

u = np.linspace(0,n,n+1)

u[0:25] = 100
u[25:] = 140

##########################################################################  
# Create the MHE GEKKO Model
mhe = GEKKO(remote=False)

mhe.time = np.linspace(0,10,11)

# Define GEKKO Parameters
mhe.Beta = mhe.Param(Beta)
mhe.Betas = [mhe.Param(Betas[i]) for i in range(len(Betas))]
mhe.Lambdas = [mhe.Param(Lambdas[i]) for i in range(len(Lambdas))]
mhe.rho_0 = mhe.Param(rho_0)
mhe.b = mhe.Param(b)
mhe.l = mhe.Param(l)
mhe.alpha_f = mhe.Param(alpha_f)
mhe.alpha_c = mhe.Param(alpha_c)
mhe.A = mhe.Param(6.314e3) # m^2
mhe.Mf = mhe.Param(122789.605) # kg
mhe.Cpf = mhe.Param(0.0002999667) # MW * s * kg^-1 * C^-1 at 700 C for UO2
mhe.J = mhe.Param(38) # MW/%
mhe.Mc = mhe.Param(1.917e5) # kg
mhe.mdot = mhe.Param(65.9e6 / 3600) # kg/s, alternative of 800 kg/s
mhe.Cpc = mhe.Param(0.00571) # MW * s * kg^-1 * C^-1 at @ 298 C for water
mhe.Cv = mhe.Param(0.0301) # flow coefficient
mhe.De = mhe.Param(0.01297) # m
mhe.Kc = mhe.Param(0.00000055) # MW/m*K
mhe.mu = mhe.Param(0.000068) # Pa*s
mhe.density = mhe.Param(1000) # kg/m^3

# Define GEKKO Fixed Value, U will be estimated
mhe.U = mhe.FV(value = U,lb = 0)
mhe.U.STATUS = 0
mhe.Tin = mhe.FV(value = Tin)

# Define GEKKO Variables
mhe.P = mhe.SV(value = 0)
mhe.Cs = [mhe.Var(Ctot*Lambdas[i]) for i in range(len(Lambdas))]
mhe.rho = mhe.Var(rho_0)
mhe.Tavg = mhe.CV(300)
mhe.Tavg.STATUS = 0
mhe.Tavg.FSTATUS = 1
mhe.Tf = mhe.SV(320)

# Define manipulated variable that will accept measurements from the MPC
mhe.u = mhe.MV(100)
mhe.u.FSTATUS = 1

# Define GEKKO equations
for i in range(len(Lambdas)):
    mhe.Equation(mhe.Cs[i].dt() == mhe.Betas[i]/mhe.l * mhe.P - mhe.Lambdas[i] * mhe.Cs[i])
mhe.Equation(mhe.P.dt() == (mhe.rho - mhe.Beta) * mhe.P / mhe.l + 
             (mhe.Lambdas[0] * mhe.Cs[0] + mhe.Lambdas[1] * mhe.Cs[1] + mhe.Lambdas[2] * mhe.Cs[2] + 
              mhe.Lambdas[3] * mhe.Cs[3] + mhe.Lambdas[4] * mhe.Cs[4] + mhe.Lambdas[5] * mhe.Cs[5]))
mhe.Equation(mhe.Tf.dt() == -mhe.U * mhe.A / (mhe.Mf * mhe.Cpf) * (mhe.Tf - mhe.Tavg) + mhe.J * mhe.P / (mhe.Mf * mhe.Cpf))
mhe.Equation(mhe.Tavg.dt()  == mhe.U * mhe.A / (mhe.Mc * mhe.Cpc) * (mhe.Tf - mhe.Tavg) - (mhe.mdot / mhe.Mc) * (mhe.Tavg - mhe.Tin))
mhe.Equation(mhe.rho == mhe.rho_0 + mhe.alpha_f * mhe.Tf + mhe.alpha_c * mhe.Tavg + mhe.b * mhe.u)

# Set GEKKO options
mhe.options.IMODE = 5
mhe.options.NODES = 2
mhe.options.SOLVER = 3

##########################################################################  
# Create MPC GEKKO Model
mpc = GEKKO(remote=False)

mpc.time = np.linspace(0,20,21)

# Create last array for use in MPC objective function
last = np.zeros(len(mpc.time))
last[5:] = 1
mpc.last = mpc.Param(last)

# Create a set point parameter for use in the objective function
# that will be updated from within the model loop
sp = np.ones(len(mpc.time))*165
mpc.SP = mpc.Param(sp)

# Create MPC parameters
mpc.Beta = mpc.Param(Beta)
mpc.Betas = [mpc.Param(Betas[i]) for i in range(len(Betas))]
mpc.Lambdas = [mpc.Param(Lambdas[i]) for i in range(len(Lambdas))]
mpc.rho_0 = mpc.Param(rho_0)
mpc.b = mpc.Param(b)
mpc.l = mpc.Param(l)
mpc.alpha_f = mpc.Param(alpha_f)
mpc.alpha_c = mpc.Param(alpha_c)
mpc.A = mpc.Param(6.314e3) # m^2
mpc.Mf = mpc.Param(122789.605) # kg
mpc.Cpf = mpc.Param(0.0002999667) # MW * s * kg^-1 * C^-1 at 700 C for UO2
mpc.J = mpc.Param(38) # MW/%
mpc.Mc = mpc.Param(1.917e5) # kg
mpc.mdot = mpc.Param(65.9e6 / 3600) # kg/s, alternative of 800 kg/s
mpc.Cpc = mpc.Param(0.00571) # MW * s * kg^-1 * C^-1 at @ 298 C for water
mpc.Cv = mpc.Param(0.0301) # flow coefficient
mpc.De = mpc.Param(0.01297) # m
mpc.Kc = mpc.Param(0.00000055) # MW/m*K
mpc.mu = mpc.Param(0.000068) # Pa*s
mpc.density = mpc.Param(1000) # kg/m^3
mpc.resupper = mpc.Param(0.9)
mpc.reslower = mpc.Param(0.4)
mpc.HP = mpc.Param(0.0)

# Create MPC fixed values
mpc.U = mpc.FV(value = U,lb = 0) # Heat transfer coefficient
mpc.U.FSTATUS = 1
mpc.Tin = mpc.FV(value = Tin) # Inlet temperature (C)

# Create MPC controlled variable (reactor power)
mpc.P = mpc.CV(value = 0)

# Create MPC manipulated variable (control rod position)
mpc.u = mpc.MV(170)
mpc.u.UPPER = 200
mpc.u.LOWER = 50
mpc.u.STATUS = 0
mpc.u.DMAX = 50

# Create other MPC variables
mpc.Cs = [mpc.Var(Ctot*Lambdas[i]) for i in range(len(Lambdas))]
mpc.rho = mpc.Var(rho_0) # Reactivity
mpc.Tavg = mpc.SV(300) # Coolant temp (C)
mpc.Tf = mpc.SV(320) # Fuel temp (C)

# Create GEKKO MPC equations
for i in range(len(Lambdas)):
    mpc.Equation(mpc.Cs[i].dt() == mpc.Betas[i]/mpc.l * mpc.P - mpc.Lambdas[i] * mpc.Cs[i])
mpc.Equation(mpc.P.dt() == (mpc.rho - mpc.Beta) * mpc.P / mpc.l + 
             (mpc.Lambdas[0] * mpc.Cs[0] + mpc.Lambdas[1] * mpc.Cs[1] + mpc.Lambdas[2] * mpc.Cs[2] + 
              mpc.Lambdas[3] * mpc.Cs[3] + mpc.Lambdas[4] * mpc.Cs[4] + mpc.Lambdas[5] * mpc.Cs[5]))
mpc.Equation(mpc.Tf.dt() == -mpc.U * mpc.A / (mpc.Mf * mpc.Cpf) * (mpc.Tf - mpc.Tavg) + mpc.J * mpc.P / (mpc.Mf * mpc.Cpf))
mpc.Equation(mpc.Tavg.dt()  == mpc.U * mpc.A / (mpc.Mc * mpc.Cpc) * (mpc.Tf - mpc.Tavg) - (mpc.mdot / mpc.Mc) * (mpc.Tavg - mpc.Tin))
mpc.Equation(mpc.rho == mpc.rho_0 + mpc.alpha_f * mpc.Tf + mpc.alpha_c * mpc.Tavg + mpc.b * mpc.u)
mpc.Equation(mpc.Tf<950.0)
mpc.Equation(mpc.Tavg<400.0)

# Set MPC options
mpc.options.IMODE = 6
mpc.options.NODES = 2
mpc.options.SOLVER = 3
mpc.options.CV_TYPE = 1
mpc.options.MAX_ITER = 1000

##########################################################################  
# Create and run the simulation loop
#Create storage arrays
Tfplot = np.zeros(n+1)
Tavgplot = np.zeros(n+1)
Pplot = np.zeros(n+1)
rhoplot = np.zeros(n+1)
Uplot = np.zeros(n+1)
time = np.linspace(0,n,n+1)

P_data = np.ones(n+1)
Tf_data = np.ones(n+1)
Tavg_data = np.ones(n+1)
rho_data = np.zeros(n+1)
U_data = np.zeros(n+1)
dRv_data = np.zeros(n+1)
Rv_data = np.zeros(n+1)
HP_data = np.zeros(n+1)
sp_store = np.zeros(n+1)

# Set initial values
P_ss = 0.0
Tf_ss = 300 # C
Tavg_ss = 294.1 # C
Tin_ss = 292 # C
Rv_data[0] = 0.75 #initial reservoir level

# initial C values
C1_ss = Ctot * Lambdas[0]
C2_ss = Ctot * Lambdas[1]
C3_ss = Ctot * Lambdas[2]
C4_ss = Ctot * Lambdas[3]
C5_ss = Ctot * Lambdas[4]
C6_ss = Ctot * Lambdas[5]

x0 = [P_ss, C1_ss, C2_ss, C3_ss, C4_ss, C5_ss, C6_ss, Tf_ss, Tavg_ss]

# Create a random multiplier for the heat transfer coefficient, U
mult1 = (0.001*np.random.randn(len(time)))
mult2 = np.linspace(0.02,0.015,len(time))

# A parameter relating outlet reservoir flow to power production
# that is set to almost zero initially but will be increased after
# control is turned on
fudge = 0.00001 

# Set up PID controller
e = np.zeros(n+1)
ie = np.zeros(n+1)
P_pid = np.zeros(n+1)
I_pid = np.zeros(n+1)
Rvsp = np.zeros(n+1)
HPadj = np.zeros(n+1)

Kp = -0.05/75.0
Kc = 1/Kp
res_setpoint = 0.5  # Initial set point, can be changed within
                    # the loop, but is only active when control = 1
                    
control = 0 # Controller switch

# Run model and control/estimation loop
print('i\tTavg\tTf\tU')
for i in range(len(power)):
    # time span for this calculation step
    ts = [time[i],time[i+1]]
    
    #run the simulation
    mult = mult1[i] + mult2[i]
    y = odeint(reactor,x0,ts,args=(u[i],mult))
    
    # store results in respective arrays
    P_data[i+1] = y[-1][0]
    Tf_data[i+1] = y[-1][7]
    Tavg_data[i+1] = y[-1][8]
    U_data[i+1] = save.U
    rho_data[i+1] = rho_0 + alpha_f * Tf_data[i+1] + alpha_c * Tavg_data[i+1] + b * u[i]
    
    # Prepare next initial condition
    x0 = [P_data[i+1],y[-1][1],y[-1][2],y[-1][3],y[-1][4],y[-1][5],y[-1][6],Tf_data[i+1],Tavg_data[i+1]]
    
    # Pass data to MHE model
    mhe.Tavg.MEAS = Tavg_data[i+1]
    mhe.u.MEAS = u[i]
    
    # Solve the MHE and MPC models in GEKKO
    mhe.solve(disp = False)  
    mpc.solve(disp = False)
    
    # Perform reservoir level calculations
    dRv_data[i] = (mpc.P.VALUE[0] - mpc.SP.VALUE[0])*fudge
    Rv_data[i+1] = Rv_data[i] + (dRv_data[i])
    
    # Run PID controller if active
    HPadj[i+1] = 0.0
    if control == 1: 
        e[i+1] = res_setpoint - Rv_data[i+1] 
        P_pid[i+1] = Kc*e[i+1]
        
        HPadj[i+1] = P_pid[i+1]
        
        if HPadj[i+1] > 20:
            HPadj[i+1] = 20
        if HPadj[i+1] < -20:
            HPadj[i+1] = -20
    
    # Store reservoir level and hydropower data
    HP = -(dRv_data[i])*2000
    HP_data[i+1] = HP
    
    # Prepare hydropower implemenation in objective function, if turned on
    mpc.HP.VALUE = HPadj[i+1]
    
    # Turn on PID reservoir control at time 80 because it's going to rain soon
    if i == 80*res:
        control = 1
    # Turn off reservoir PID control
    if i == 122*res:
        control = 0.0
    
    # Add to reservoir to simulate rainfall
    if i > 122*res and i < 140*res:
        Rv_data[i+1] = Rv_data[i+1] + 0.01/res
        
    # Save U value for plotting and update value in MPC model
    Uplot[i+1] = mhe.U.NEWVAL
    mpc.U.MEAS = Uplot[i+1]
    
    # Turn on GEKKO MHE at time 25 hrs
    if i > 25*res:
        mhe.Tavg.STATUS = 1
        mhe.U.STATUS = 1
        
    # Turn on GEKKO MPC at time 35 hrs
    if i > 35*res:
        mpc.P.FSTATUS = 0
        mpc.P.STATUS = 0
        mpc.u.STATUS = 1
        
        # Update MV with new value for use in ODEINT and MHE in next cycle
        u[i+1] = mpc.u.NEWVAL
        
        # Feed the set point into the GEKKO MPC model
        sp_store[i] = mpc.SP.VALUE[-1]
        sp = np.ones(len(mpc.time))*power[i] # power[i] is the current demand=
        mpc.SP.VALUE = sp

        # Update the power to reservoir output relationship now that
        # the MHE and MPC are running
        fudge = 0.0005
        
        # MPC objective function. It ignores the first 5 values of the
        # power array because there is characteristic spiking that occurs
        # in nuclear systems that makes solving difficult if those values
        # are included.
        mpc.Obj((mpc.P*mpc.last-(mpc.SP - mpc.HP))**2)

    # Print some relevant information every 5 time steps
    if i%5 == 0:
        print(i,'\t',round(Tavg_data[i],1),'\t',round(Tf_data[i],1),'\t',round(U_data[i],5),'\t',round(mpc.P.VALUE[0],2))

    # Plot the data, update the plots every 10 time steps
    if i%10 == 0:
        plt.figure(1)
        plt.clf()
        
        plt.subplot(4,1,1)
        plt.plot(time[1:i]/res,Tf_data[1:i],'b--',label='Tf ODE')
        plt.plot(time[1:i]/res,Tavg_data[1:i],'g--',label='Tavg ODE')
        plt.ylabel('Temp (C)')
        plt.legend()
        
        plt.subplot(4,1,2)
        plt.plot(time[1:i]/res,u[1:i],'r--',label='Control Rod Step')
        plt.ylabel('Steps')
        plt.legend()
        
        plt.subplot(4,1,3)
        plt.plot(time[1:i]/res,rho_data[1:i],'r--',label='Reactivity ODE')
        plt.ylabel('Reactivity')
        plt.legend()
        
        plt.subplot(4,1,4)
        plt.plot(time[1:i]/res,Uplot[1:i],'-',label='MHE')
        plt.plot(time[1:i]/res,U_data[1:i],'.',label='Actual')
        plt.legend()
        plt.ylabel('U (MW/m^2/K')
        plt.xlabel('Time (hr)')
        plt.pause(0.000001)
        
        plt.figure(2)
        plt.clf()
        plt.subplot(2,1,1)
        plt.plot(time[1:i]/res,P_data[1:i],'r--',label='Nuclear Supply')
        plt.plot((time[1:i])/res,power[1:i],'c--',label='Nuclear Demand')
        plt.plot((time[1:i])/res,solar_power[1:i],'y--',label='Solar Supply')
        plt.plot((time[1:i])/res,wind_power[1:i],'g--',label='Wind Supply')
        plt.plot((time[1:i])/res,HP_data[1:i],'b--',label='Hydro Supply')
        plt.plot((time[1:i])/res,tot_power[1:i],'k--',label='Total Demand')
        plt.ylabel('Power')
        plt.legend()
        
        plt.subplot(2,1,2)
        plt.plot(time[1:i]/res,Rv_data[1:i],'b--',label='Reservoir Level')
        plt.legend()
        plt.ylabel('% Full')
        plt.xlabel('Time (hr)')
        plt.ylim(0,1)
        plt.pause(0.000001)
        
##########################################################################  
# Store the data in an excel file
dictionary = {'time': time[1:]/res,
              'fuel temp': Tf_data[1:],
              'water temp': Tavg_data[1:],
              'control rods': u[1:],
              'rho': rho_data[1:],
              'U MHE': Uplot[1:],
              'U actual': U_data[1:],
              'nuclear supply': P_data[1:],
              'nuclear demand': power,
              'solar supply': solar_power,
              'wind supply': wind_power,
              'hydro supply': HP_data[1:],
              'total demand': tot_power,
              'reservoir fill': Rv_data[1:]}      

df = pd.DataFrame.from_dict(dictionary)

df.to_excel('DOFinalData.xlsx')